/* $Id: auto.c,v 1.26 1998/07/20 21:12:52 ericb Exp $ */
/* Copyright (C) 1994 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* This file contains the auto-zero and auto-range functions. */

#include "sema.h"

/*#define	AR_DEBUG*/
#ifdef	AR_DEBUG
#define	AR_PRINT(_s)	(void) printf _s
#else
#define	AR_PRINT(_s)	/*(void) printf _s*/
#endif

/*
 *********************************************************************
 Given an input range, figure out the next range setting, either up or
 down.  This assumes the valid ranges are 1-2-5 spaced.
 *********************************************************************
 */
static FLOATSIZ32
range_next(FLOATSIZ32 range, int up)
{
    float  tmp;

    if (range <= 0)
	range = 0.005F;

    tmp = range;
    while (tmp < 1)
	tmp *= 10.0F;
    while (tmp >= 10)
	tmp *= 0.1F;

    if (up)
    {
	if (tmp > 1.5 && tmp < 3)
	    return range * 2.5F;
	else
	    return range * 2.0F;
    }
    else
    {
	if (tmp > 4 && tmp < 7)
	    return range * 0.4F;
	else
	    return range * 0.5F;
    }
}

/*
 *********************************************************************
 Update the range of one channel.  Return positive if the range
 changed successfully, zero if it can't be changed, negative if
 error.
 *********************************************************************
 */
static SHORTSIZ16
range_update(E1432ID hw, SHORTSIZ16 ID, int up, int range_type)
{
    SHORTSIZ16 error;
    FLOATSIZ32 range, new_range;
    int     save_print_errors;

    switch (range_type)
    {
    case E1432_INPUT_MODE_CHARGE:
	error = e1432_get_range_charge(hw, ID, &range);
	break;
    case E1432_INPUT_MODE_MIC:
    case E1432_INPUT_MODE_MIC_200V:
	error = e1432_get_range_mike(hw, ID, &range);
	break;
    default:
	error = e1432_get_range(hw, ID, &range);
	break;
    }
    if (error)
	return error;

    new_range = range_next(range, up);

    AR_PRINT(("ID 0x%02x up %d range %6.3f -> %6.3f ->",
	      ID, up, range, new_range));

    save_print_errors = i1432_print_errors;
    i1432_print_errors = 0;
    switch (range_type)
    {
    case E1432_INPUT_MODE_CHARGE:
	error = e1432_set_range_charge(hw, ID, new_range);
	break;
    case E1432_INPUT_MODE_MIC:
    case E1432_INPUT_MODE_MIC_200V:
	error = e1432_set_range_mike(hw, ID, new_range);
	break;
    default:
	error = e1432_set_range(hw, ID, new_range);
	break;
    }
    i1432_print_errors = save_print_errors;
    if (error < 0 && error != ERR1432_RANGE_OUT_OF_LIMITS)
	return error;

    switch (range_type)
    {
    case E1432_INPUT_MODE_CHARGE:
	error = e1432_get_range_charge(hw, ID, &new_range);
	break;
    case E1432_INPUT_MODE_MIC:
    case E1432_INPUT_MODE_MIC_200V:
	error = e1432_get_range_mike(hw, ID, &new_range);
	break;
    default:
	error = e1432_get_range(hw, ID, &new_range);
	break;
    }
    if (error)
	return error;

    AR_PRINT((" %6.3f\n", new_range));

    if (new_range == range)
	return 0;		/* Already at min or max range */

    return 1;
}

/*
 *********************************************************************
 Check each channel for overload/halfrange, and update the status
 array.
 *********************************************************************
 */
static SHORTSIZ16
do_one_scan(E1432ID hw, E1432_GROUP_LIST_NODE *gn, int *status)
{
    E1432_CHAN_LIST_NODE *cn;
    SHORTSIZ16 error, diff, half;
    int *ptr;

    /* Check overload status of each channel */
    cn = gn->chanlist;
    ptr = status;
    while (cn != NULL)
    {
	/* Currently ignoring any common-mode overloads */
	error = e1432_check_overloads(hw, cn->chanID, NULL,
				      NULL, &diff, &half);
	if (error)
	    return error;

	if (diff)
	    *ptr = 1;
	else if (half && *ptr < 0)
	    *ptr = 0;

	cn = cn->next;
	ptr++;
    }
    return 0;
}

/*
 *********************************************************************
 A positive return value means that all channels are currently
 auto-ranged correctly.  Negative means error, zero means we need to
 do another pass.
 *********************************************************************
 */
static SHORTSIZ16
do_one_pass(E1432ID hw, SHORTSIZ16 ID, E1432_GROUP_LIST_NODE *gn,
	    FLOATSIZ64 meas_time, int prevent_range_down,
	    int *status, int *range_type, int extra_wait)
{
    E1432_CHAN_LIST_NODE *cn;
    FLOATSIZ64 start, current;
    SHORTSIZ16 error, ok, ar_mode;
    int    *ptr, *tptr;
    int     type, count;

    /* Set status array to -1 indicating underrange */
    cn = gn->chanlist;
    ptr = status;
    while (cn != NULL)
    {
	*ptr++ = -1;
	cn = cn->next;
    }

    if (extra_wait)
	/* We are using Charge or Mike BoBs, which take longer to
	   settle, so wait here before doing this pass.  We could put
	   this into the module firmware in the SETTLE command, but
	   then we would have to wait if any channel was hooked to a
	   BoB, even if that channel was not part of the
	   auto-range.  This was unreliable at 0.04 seconds, and
	   appeared completely reliable at 0.1 seconds.  Pad that to
	   be sure. */
	i1432_pause(0.15);

    /* Wait for hardware to settle, which also clears overload info */
    error = i1432_set_mod(hw, ID, -1, E1432_CMD_SETTLE, 0);
    if (error)
	return error;

    /* Check for block available, which may flush a lost scan, which
       may allow the measurement state machine to loop, which may
       allow another source burst, which may allow us to autorange
       with a bursting source. */
    error = e1432_block_available(hw, ID);
    /* Return value can be positive, which is not an error */
    if (error < 0)
	return error;

    /* Monitor overload/halfrange status for meas_time seconds */
    i1432_get_time(&start);
    do
    {
	/* Check overload status of each channel */
	error = do_one_scan(hw, gn, status);
	if (error)
	    return error;
	i1432_get_time(&current);
    }
    while (current - start < meas_time);

    /* Force overload/halfrange status update */
    error = i1432_set_mod(hw, ID, -1, E1432_CMD_OVLD_UPDATE, 0);
    if (error)
	return error;

    /* Check overload/halfrange one last time */
    error = do_one_scan(hw, gn, status);
    if (error)
	return error;

    /* Update ranges */
    ok = 1;
    cn = gn->chanlist;
    ptr = status;
    tptr = range_type;
    while (cn != NULL)
    {
	/* Get the channel type and count, to get auto-range mode */
	type = (cn->chanID & E1432_CHAN_TYPE_MASK) >> E1432_CHAN_TYPE_SHIFT;
	count = (cn->chanID & E1432_CHAN_MASK) - 1;
	ar_mode = i1432_chan_list[type][count].auto_range_mode;
	if (prevent_range_down)
	    ar_mode &= ~E1432_AUTO_RANGE_MODE_DOWN;

	if ((*ptr > 0 && (ar_mode & E1432_AUTO_RANGE_MODE_UP) != 0) ||
	    (*ptr < 0 && (ar_mode & E1432_AUTO_RANGE_MODE_DOWN) != 0))
	{
	    error = range_update(hw, cn->chanID, *ptr > 0, *tptr);
	    if (error < 0)	/* Error */
		return error;
	    if (error > 0)	/* Not at range min or max */
		ok = 0;
	}
#ifdef	AR_DEBUG
	else
	    AR_PRINT(("ID 0x%02x done\n", cn->chanID));
#endif
	cn = cn->next;
	ptr++;
	tptr++;
    }

    return ok;
}

static SHORTSIZ16
range_type_init(E1432ID hw, E1432_GROUP_LIST_NODE *gn,
		int *range_type, int *extra_wait)
{
    E1432_CHAN_LIST_NODE *cn;
    SHORTSIZ16 error, mode, ih;

    *extra_wait = 0;

    cn = gn->chanlist;
    while (cn != NULL)
    {
	error = e1432_get_input_mode(hw, cn->chanID, &mode);
	if (error)
	    return error;

	/* Input high CALIN overrides CHARGE or MIKE settings and
	   should be treated like VOLT. */
	if (mode != E1432_INPUT_MODE_VOLT &&
	    mode != E1432_INPUT_MODE_ICP)
	{
	    error = e1432_get_input_high(hw, cn->chanID, &ih);
	    if (error)
		return error;
	    if (ih == E1432_INPUT_HIGH_CALIN)
		mode = E1432_INPUT_MODE_VOLT;
	    else
		*extra_wait = 1;
	}
	*range_type++ = mode;

	cn = cn->next;
    }

    return 0;
}

/*
 *********************************************************************
 Do the work involved with auto-range, assuming that the caller has
 already ensured that a measurement is running.  The caller also must
 set up <gn> and pass in the <count> of the number of channels.
 *********************************************************************
 */
static SHORTSIZ16
internal_auto_range(E1432ID hw, SHORTSIZ16 ID, FLOATSIZ64 meas_time,
		    E1432_GROUP_LIST_NODE *gn, int count)
{
    SHORTSIZ16 error;
    FLOATSIZ32 span;
    LONGSIZ32 bs;
    int     i, extra_wait;
    int    *status, *range_type;

    /* Possibly calculate how long to auto-range */
    if (meas_time == 0 && count > 0)
    {
	/* Use span and blocksize of first channel, in case multiple
	   channels are not all at the same span and blocksize */
	error = e1432_get_span(hw, gn->chanlist->chanID, &span);
	if (error)
	    return error;
	error = e1432_get_blocksize(hw, gn->chanlist->chanID, &bs);
	if (error)
	    return error;
	meas_time = bs * 0.1 / span;
	AR_PRINT(("Auto-range meas time set to %g\n", meas_time));
    }

    /* Allocate list of module status for use by do_one_pass */
    status = malloc(count * sizeof *status);
    if (status == NULL)
	return i1432_id_print_error(hw, ID,
				    ERR1432_AUTO_RANGE_FAILURE);

    /* Allocate list of input range types for use by do_one_pass */
    range_type = malloc(count * sizeof *range_type);
    if (range_type == NULL)
	return i1432_id_print_error(hw, ID,
				    ERR1432_AUTO_RANGE_FAILURE);

    /* Initialize range type array for use by do_one_pass */
    error = range_type_init(hw, gn, range_type, &extra_wait);
    if (error)
	return error;

    for (i = 0; i < 30; i++)
    {
	AR_PRINT(("Auto-range pass %d\n", i));
	/* After 20 passes, set prevent_range_down parameter so that
	   an out-of-band overload won't make us toggle forever. */
	error = do_one_pass(hw, ID, gn, meas_time, i > 20,
			    status, range_type, extra_wait);
	if (error)
	    break;
    }
    AR_PRINT(("Auto-range done: %d\n", error));

    free(status);

    if (error < 0)		/* Error */
	return error;
    if (error > 0)		/* Successful */
	return 0;

    return i1432_id_print_error(hw, ID, ERR1432_AUTO_RANGE_FAILURE);
}

/*
 *********************************************************************
 Auto-range one channel, or all channels in a group, <ID>, checking
 input for <meas_time> seconds or until an overload occurs.  If an
 overload occurs, the range is bumped up and is checked for another
 <meas_time> seconds.

 If meas_time = 0.0, estimate time to do auto-range.  We use the same
 formula used by Geoduck and by E1431:

	time = blocksize * 0.1 / span.

 Returns negative error number if error, otherwise returns 0.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_auto_range(E1432ID hw, SHORTSIZ16 ID, FLOATSIZ64 meas_time)
{
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn;
    E1432_GROUP_LIST_NODE dummy_group;
    E1432_CHAN_LIST_NODE dummy_chan;
    E1432_MODULE_LIST_NODE *dummy_mn;
    FLOATSIZ32 cf;
    SHORTSIZ16 error, meas_state, return_save, chan_id;
    SHORTSIZ16 *data_port_save, *irq_mask_save, *src_active_save, *p;
    int     count, mod, chan;

    TRACE_PRINTF(0, ("e1432_auto_range(0x%p, %d, %g)\n",
		     hw, ID, meas_time));

    return_save = 0;
    data_port_save = NULL;
    irq_mask_save = NULL;
    src_active_save = NULL;

    if (ID < 0)
    {
	/* Auto-ranging a group, count the number of channels */
	gn = i1432_get_group_node(hw, ID);
	cn = gn->chanlist;
	count = 0;
	while (cn != NULL)
	{
	    count++;
	    cn = cn->next;
	}
    }
    else
    {
	/* Auto-ranging just one channel, fake like it's a group */
	dummy_chan.chanID = ID;
	dummy_chan.next = NULL;
	dummy_group.chanlist = &dummy_chan;
	error = i1432_get_module_from_chan(hw, ID, &dummy_mn);
	if (error)
	    goto cleanup;
	dummy_group.modlist = &dummy_mn;
	dummy_group.modcount = 1;
	gn = &dummy_group;
	count = 1;
    }

    /* Check for clock frequency too high */
    for (mod = 0; mod < gn->modcount; mod++)
    {
	error = e1432_get_clock_freq(hw, gn->modlist[mod]->chan_id,
				     &cf);
	if (error)
	    return error;
	if (cf > 128000.0)
	    return i1432_id_print_error(hw, ID,
					ERR1432_AUTO_RANGE_VS_CLOCK_F);
    }

    /* Make sure a measurement is running */
    error = e1432_get_meas_state(hw, ID, &meas_state);
    if (error)
	goto cleanup;
    if (meas_state == E1432_MEAS_STATE_TESTED)
    {
	/* Measurement not running.  Start one, but first ensure that
	   we don't send data to local bus, don't start any
	   sources, and don't interrupt for any reason. */

	/* Allocate list of saved data port settings */
	data_port_save = malloc(gn->modcount * sizeof *data_port_save);
	if (data_port_save == NULL)
	{
	    error = i1432_id_print_error(hw, ID,
					 ERR1432_AUTO_RANGE_FAILURE);
	    goto cleanup;
	}
	/* Allocate list of saved irq mask settings */
	irq_mask_save = malloc(gn->modcount * sizeof *irq_mask_save);
	if (irq_mask_save == NULL)
	{
	    error = i1432_id_print_error(hw, ID,
					 ERR1432_AUTO_RANGE_FAILURE);
	    goto cleanup;
	}
	/* Allocate list of saved src active settings.  We can have
	   up to one source channel per SCA connector. */
	src_active_save = malloc(gn->modcount * E1432_SCAS *
				 sizeof *src_active_save);
	if (src_active_save == NULL)
	{
	    error = i1432_id_print_error(hw, ID,
					 ERR1432_AUTO_RANGE_FAILURE);
	    goto cleanup;
	}

	/* Save data port settings, then set to VME */
	p = data_port_save;
	for (mod = 0; mod < gn->modcount; mod++)
	{
	    error = e1432_get_data_port(hw, gn->modlist[mod]->chan_id,
					p++);
	    if (error)
		goto cleanup;
	    error = e1432_set_data_port(hw, gn->modlist[mod]->chan_id,
					E1432_SEND_PORT_VME);
	    if (error)
		goto cleanup;
	}

	/* Save irq mask settings, then set to 0 */
	p = irq_mask_save;
	for (mod = 0; mod < gn->modcount; mod++)
	{
	    error = e1432_get_interrupt_mask(hw, gn->modlist[mod]->chan_id,
					     p++);
	    if (error)
		goto cleanup;
	    error = e1432_set_interrupt_mask(hw, gn->modlist[mod]->chan_id,
					     0);
	    if (error)
		goto cleanup;
	}

	/* Save src active settings, then set to inactive.  We must do
	   this for all sources in any module in this group, not just
	   the channels in the group, because e1432_init_measure
	   starts all active channels in each module. */
	p = src_active_save;
	/* Scan each module in the group */
	for (mod = 0; mod < gn->modcount; mod++)
	    /* Scan every source channel in existance */
	    for (chan = 0;
		 chan < i1432_chan_count[E1432_CHAN_TYPE_SOURCE];
		 chan++)
		/* Is this source channel in this module? */
		if (i1432_chan_list[E1432_CHAN_TYPE_SOURCE][chan].mn
		    == gn->modlist[mod])
		{
		    /* Source channel is in the module */
		    chan_id = (SHORTSIZ16)
			(chan + 1 + (E1432_CHAN_TYPE_SOURCE <<
				     E1432_CHAN_TYPE_SHIFT));
		    /* Get current active setting */
		    error = e1432_get_active(hw, chan_id, p++);
		    if (error)
			goto cleanup;
		    error = e1432_set_active(hw, chan_id,
					     E1432_CHANNEL_OFF);
		    if (error)
			goto cleanup;
		}

	/* Start auto-range measurement */
	error = e1432_init_measure(hw, ID);
	if (error)
	    goto cleanup;
    }

    /* Do the auto-range */
    return_save = internal_auto_range(hw, ID, meas_time, gn, count);

    if (meas_state == E1432_MEAS_STATE_TESTED)
    {
	/* Kill the auto-range measurement that we started earlier */
	error = e1432_reset_measure(hw, ID);
	if (error)
	    goto cleanup;

	/* Restore data port settings */
	p = data_port_save;
	for (mod = 0; mod < gn->modcount; mod++)
	{
	    error = e1432_set_data_port(hw, gn->modlist[mod]->chan_id,
					*p++);
	    if (error)
		goto cleanup;
	}

	/* Restore interrupt mask settings */
	p = irq_mask_save;
	for (mod = 0; mod < gn->modcount; mod++)
	{
	    error = e1432_set_interrupt_mask(hw, gn->modlist[mod]->chan_id,
					     *p++);
	    if (error)
		goto cleanup;
	}

	/* Restore src active settings */
	p = src_active_save;
	/* Scan each module in the group */
	for (mod = 0; mod < gn->modcount; mod++)
	    /* Scan every source channel in existance */
	    for (chan = 0;
		 chan < i1432_chan_count[E1432_CHAN_TYPE_SOURCE];
		 chan++)
		/* Is this source channel in this module? */
		if (i1432_chan_list[E1432_CHAN_TYPE_SOURCE][chan].mn
		    == gn->modlist[mod])
		{
		    /* Source channel is in the module */
		    chan_id = (SHORTSIZ16)
			(chan + 1 + (E1432_CHAN_TYPE_SOURCE <<
				     E1432_CHAN_TYPE_SHIFT));
		    error = e1432_set_active(hw, chan_id, *p++);
		    if (error)
			goto cleanup;
		}
    }

cleanup:
    if (return_save == 0)
	return_save = error;
    free(data_port_save);
    free(irq_mask_save);

    return return_save;
}

SHORTSIZ16 EXPORT
e1432_get_auto_range_mode(E1432ID hw, SHORTSIZ16 ID, SHORTSIZ16 *mode)
{
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn;
    SHORTSIZ16 error, temp;
    int     first, type, count;

    TRACE_PRINTF(0, ("e1432_get_auto_range_mode(0x%p, %d, 0x%p)\n",
		     hw, ID, mode));

    /* Check for valid id */
    error = i1432_checkID(hw, ID);
    if (error)
	return error;

    /* Check if channel or group */
    if (ID < 0)
    {
	/* Iterate through group making sure that all parameters match */
	gn = i1432_get_group_node(hw, ID);
	cn = gn->chanlist;
	first = 1;
	while (cn != NULL)
	{
	    error = e1432_get_auto_range_mode(hw, cn->chanID, &temp);
	    if (error)
		return error;
	    if (first)
	    {
		*mode = temp;
		first = 0;
	    }
	    if (temp != *mode)
		return i1432_print_error(ERR1432_PARAMETER_UNEQUAL);
	    cn = cn->next;
	}
    }
    else
    {
	/* Get lone channel */
	type = (ID & E1432_CHAN_TYPE_MASK) >> E1432_CHAN_TYPE_SHIFT;
	count = (ID & E1432_CHAN_MASK) - 1;
	*mode = i1432_chan_list[type][count].auto_range_mode;
    }

    return 0;
}

/*
 *********************************************************************
 Set auto range mode.  This parameter is currently stored in the host
 rather than the module, which is different from almost all other
 parameters.  This might not be a good idea, but it's convenient and
 it's hidden behind the API so we can always change it later.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_set_auto_range_mode(E1432ID hw, SHORTSIZ16 ID, SHORTSIZ16 mode)
{
    E1432_GROUP_LIST_NODE *gn;
    E1432_CHAN_LIST_NODE *cn;
    SHORTSIZ16 error;
    int     type, count;

    TRACE_PRINTF(1, ("e1432_set_auto_range_mode(0x%p, %d, %d)\n",
		     hw, ID, mode));

    /* Check for valid id */
    error = i1432_checkID(hw, ID);
    if (error)
	return error;

    /* Limit check on parameter */
    if ((mode & ~(E1432_AUTO_RANGE_MODE_UP |
		  E1432_AUTO_RANGE_MODE_DOWN)) != 0)
	return i1432_print_error(ERR1432_ILLEGAL_AUTO_RANGE_MODE);

    /* Check if channel or group */
    if (ID < 0)
    {
	/* Iterate through group */
	gn = i1432_get_group_node(hw, ID);
	cn = gn->chanlist;
	while (cn != NULL)
	{
	    error = e1432_set_auto_range_mode(hw, cn->chanID, mode);
	    if (error)
		return error;
	    cn = cn->next;
	}
    }
    else
    {
	/* Set lone channel */
	type = (ID & E1432_CHAN_TYPE_MASK) >> E1432_CHAN_TYPE_SHIFT;
	count = (ID & E1432_CHAN_MASK) - 1;
	i1432_chan_list[type][count].auto_range_mode = mode;
    }
    return 0;
}

/*
 *********************************************************************
 Auto zero a group.  Returns negative error number if error, otherwise
 returns 0.
 *********************************************************************
 */
SHORTSIZ16 EXPORT
e1432_auto_zero(E1432ID hw, SHORTSIZ16 ID)
{
    E1432_MODULE_LIST_NODE *mn;
    E1432_GROUP_LIST_NODE *gn;
    SHORTSIZ16 error;
    int     mod;

    TRACE_PRINTF(0, ("e1432_auto_zero(0x%p, %d)\n", hw, ID));

    /* Do a reset measure before trying to auto-zero.  This is not
       necessary when only one module is involved, nor is it necessary
       if there are only inputs in the system.  But there are corner
       cases in multi-module systems where an auto-zero in one module
       might cause active sources in other modules to start outputting
       a signal.  We want to avoid that if at all possible, because it
       is almost certainly unexpected.

       It used to be that e1432_reset_measure was not sufficient for
       this.  But now e1432_reset_measure leaves the measurement state
       frozen in the TESTED state, so e1432_reset_measure will do the
       job. */
    error = e1432_reset_measure(hw, ID);
    if (error)
	return error;

    /* Set auto-zero flag for each channel in group */
    error = i1432_set_chan(hw, ID, -1, E1432_CCMD_SET_AUTO_ZERO,
			   E1432_AUTO_ZERO_ON);
    if (error)
	return error;

    /* Check if channel or group */
    if (ID < 0)
    {
	/* Autozero each module in group */
	gn = i1432_get_group_node(hw, ID);
	for (mod = 0; mod < gn->modcount; mod++)
	{
	    mn = gn->modlist[mod];
	    error = i1432_write_cmd0(hw,
				     i1432_get_chan_from_module(mn),
				     E1432_CMD_AUTO_ZERO);
	    if (error)
		return error;
	}
    }
    else
    {
	/* Autozero lone channel */
	error = i1432_write_cmd0(hw, ID, E1432_CMD_AUTO_ZERO);
	if (error)
	    return error;
    }

    /* Clear auto-zero flag for each channel in group */
    error = i1432_set_chan(hw, ID, -1, E1432_CCMD_SET_AUTO_ZERO,
			   E1432_AUTO_ZERO_OFF);
    if (error)
	return error;

    return 0;
}
